/**
 *
 * \file        autodiscovery_device.c
 *
 * \brief       Implements the "device" side of Crestron's proprietary
 *              protocol for automatic discovery of other Crestron devices
 *              over ethernet/TCP/IP.
 *
 * \author      Pete McCormick
 *
 * \date        11/8/2005
 *
 * \note        The point of all this is to avoid having to manually configure things
 *              on both the device side and the control system side.
 *
 *              This is intended to be a generic, cross-platform module.
 *
 *              Packets are sent over UDP in network (big-endian) order.
 *              Sorry Intel!
 *
 * \note        This file is from TLIB archive located at \\crestdata2a\Swarchive\cajita\libs\
 *
 */

////////////////////////////////////////////////////////////////////////////////

#include "autodiscovery.h"              // only include this file!
#include "console.h"
#include "errors.h"
#include "dm_hardware.h"
#ifdef DM_V24
#define ErrorPrintf V24ErroLog
#else
#define ErrorPrintf DmConsolePrintf
#endif

#define AUTODEBUG

extern "C"
  {
    #include "memorymanager.h"
  }



////////////////////////////////////////////////////////////////////////////////

// structure for a hosts entry
typedef struct
{
    char HostName[MAX_DNS_NAME_LENGTH];
    UINT32 IP_Address;
    UINT32 msec;
} AUTODISCHOST;

// overall autodiscovery structure
typedef struct
{
    UINT8 debugLevel;
    UINT8 enabled;
    UINT8 RespondOnNoIpTableQuery;
    UINT8 blinking;
#ifdef STATS
    UINT32 rcvPktCnt;
    UINT32 queryCnt;
    UINT32 hostQueryCnt;
    UINT32 hostRespCnt;
#endif // STATS
    UINT32 srcIpAddr;
    AUTODISCHOST hostsEntry[AUTODISC_MAX_HOSTS];
} AUTODISCOVERY;

////////////////////////////////////////////////////////////////////////////////

static AUTODISCOVERY Autodiscovery;

#ifdef SERIES_3_CONTROL
void (*g_pFuncUpdateDownstreamIpt)(void) = NULL;
#endif

////////////////////////////////////////////////////////////////////////////////

/**
 * \author      Pete McCormick
 *
 * \date        2/23/2006
 *
 * \return      INT32
 *
 * \retval      0: success
 * \retval      <0: failure
 *
 * \brief       Function to resolve a name to an ip address, trying
 *              regular DNS first, then our own autodiscovery.
 *
 * \param       HostName
 * \param       pIpAddr
 *
 * \note        HostName is converted to UPPERCASE before searching.
 *
 */
INT32 AutodiscoveryGetHostByName(char * HostName, UINT32 * pIpAddr)
{
    UINT32 i;
    AUTODISCHOST * pEntry;
    //char tempHostname[MAX_DNS_NAME_LENGTH];
    char *tempHostname;
    CIP_HOSTNAME_QUERY_MSG * hostnameQuery;
    UINT32 retries;
    UINT32 msec;

    if (!MemMgr || (tempHostname = (char *)MemMgr->GetBlock(MAX_DNS_NAME_LENGTH)) == 0)
      {
        DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_ETHERNET, ERR_ETH_AUTODISC_MEMALLOCFAILED,0);
        return -1;
      } // if (!MemMgr || (tempHostname = (char *)MemMgr->GetBlock(MAX_DNS_NAME_LENGTH)) == 0)

    if (!MemMgr || (hostnameQuery = (CIP_HOSTNAME_QUERY_MSG *)MemMgr->GetBlock(sizeof(CIP_HOSTNAME_QUERY_MSG))) == 0)
      {
        DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_ETHERNET, ERR_ETH_AUTODISC_MEMALLOCFAILED,1);
        MemMgr->FreeBlock((unsigned char *)tempHostname);
        return -1;
      } // if (!MemMgr || (hostnameQuery = (char *)MemMgr->GetBlock(sizeof(CIP_HOSTNAME_QUERY_MSG))) == 0)



    strncpy(tempHostname, HostName, MAX_DNS_NAME_LENGTH);
    LocalConvertEntireStringToUpper(tempHostname);

// Not neccessary for products with DNS/AutoDiscovery support
#ifndef EXTENDED_ETHERNET_SUPPORT
    // try regular DNS, only if the hostname is a fully-qualified domain name
    if(strstr(tempHostname,"."))
    {
        if(OSGetHostByName(tempHostname, pIpAddr) == 0)
        {
            MemMgr->FreeBlock((unsigned char *)tempHostname);
            MemMgr->FreeBlock((unsigned char *)hostnameQuery);
            return 0;
        }
    }
#endif

    msec = HwGetMsec();

    // time out existing entries first
    for(i=0, pEntry = &Autodiscovery.hostsEntry[0];
        i<AUTODISC_MAX_HOSTS;
        i++, pEntry++)
    {
        if(!pEntry->HostName[0])
        {
            continue;
        }
        if(msec - pEntry->msec > AUTODISC_HOSTS_TTL_MSEC)
        {
            // entry has expired, remove it
            memset(pEntry, 0, sizeof(*pEntry));
        }
    }

    // otherwise try our DNS...
    for(retries=0; retries<AUTODISC_MAX_RETRIES; retries++)
    {
        // now do local lookup
        for(i=0, pEntry = &Autodiscovery.hostsEntry[0];
            i<AUTODISC_MAX_HOSTS;
            i++, pEntry++)
        {
            if(!pEntry->HostName[0])
            {
                continue;
            }
            if(strcmp(tempHostname, pEntry->HostName) == 0)
            {
                *pIpAddr = pEntry->IP_Address;
                MemMgr->FreeBlock((unsigned char *)tempHostname);
                MemMgr->FreeBlock((unsigned char *)hostnameQuery);
                return 0;
            }
        }

        // no matching entry, query
        memset(hostnameQuery, 0, sizeof(CIP_HOSTNAME_QUERY_MSG));
        hostnameQuery->Header.PacketType = CIP_HOSTNAME_QUERY;
        hostnameQuery->Header.Flags = NATIVE2BE16(hostnameQuery->Header.Flags);
        hostnameQuery->Header.DataLength = (UINT16)NATIVE2BE16(sizeof(CIP_HOSTNAME_QUERY_MSG) - sizeof(CIP_HEADER_INFO));
        _strncpy(hostnameQuery->HostName, tempHostname, MAX_DNS_NAME_LENGTH);

        // reply will be a broadcast also
        AutodiscoverySend(hostnameQuery, sizeof(CIP_HOSTNAME_QUERY_MSG), 0xffffffff);

        AutodiscoveryWaitForHostnameResp(AUTODISCOVERY_GET_HOST_MSEC);
    }

    MemMgr->FreeBlock((unsigned char *)tempHostname);
    MemMgr->FreeBlock((unsigned char *)hostnameQuery);
    return -1;
}

/**
 * \author      Pete McCormick
 *
 * \date        2/23/2006
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Send an (unsolicited) hostname response message.
 *              Used to announce ourselves when we boot up.
 *
 * \param       void
 *
 */
void AutodiscoverySendHostnameResp(void)
{
    char * myHostname;
    CIP_HOSTNAME_RESPONSE_MSG * hostnameResp;

    if (!MemMgr || (myHostname = (char *)MemMgr->GetBlock(MAX_DNS_NAME_LENGTH)) == 0)
      {
        DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_ETHERNET, ERR_ETH_AUTODISC_MEMALLOCFAILED,2);
        return;
      } // if (!MemMgr || (myHostname = (char *)MemMgr->GetBlock(MAX_DNS_NAME_LENGTH)) == 0)

    if (!MemMgr || (hostnameResp = (CIP_HOSTNAME_RESPONSE_MSG *)MemMgr->GetBlock(sizeof(CIP_HOSTNAME_RESPONSE_MSG))) == 0)
      {
        DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_ETHERNET, ERR_ETH_AUTODISC_MEMALLOCFAILED,3);
        MemMgr->FreeBlock((unsigned char *)myHostname);
        return;
      } // if (!MemMgr || (hostnameResp = (char *)MemMgr->GetBlock(sizeof(CIP_HOSTNAME_RESPONSE_MSG))) == 0)

    NetGetHostname(myHostname, MAX_DNS_NAME_LENGTH);
    LocalConvertEntireStringToUpper(myHostname);

    memset(hostnameResp, 0, sizeof(CIP_HOSTNAME_RESPONSE_MSG));

    hostnameResp->Header.PacketType = CIP_HOSTNAME_RESPONSE;
    hostnameResp->Header.Flags = NATIVE2BE16(hostnameResp->Header.Flags);
    hostnameResp->Header.DataLength = (UINT16)NATIVE2BE16(sizeof(CIP_HOSTNAME_RESPONSE_MSG) - sizeof(CIP_HEADER_INFO));

    hostnameResp->IP_Address = NetGetIPAddress();
    _strncpy(hostnameResp->HostName, myHostname, MAX_DNS_NAME_LENGTH);

    AutodiscoverySend(hostnameResp, sizeof(CIP_HOSTNAME_RESPONSE_MSG), 0xffffffff);

    MemMgr->FreeBlock((unsigned char *)myHostname);
    MemMgr->FreeBlock((unsigned char *)hostnameResp);
}

/**
 * \author      Pete McCormick
 *
 * \date        02/21/06
 *
 * \return      BOOL
 *
 * \retval      1: autodiscovery is enabled
 * \retval      0: autodiscovery is disabled
 *
 * \brief       Returns whether or not autodiscovery is enabled
 *
 * \param       void
 *
 * \note
 *
 */
BOOL AutodiscoveryIsEnabled(void)
{
    return Autodiscovery.enabled;
}

/**
 * \author      Pete McCormick
 *
 * \date        02/21/06
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Start to blink the autodiscovery LED.
 *
 * \param       void
 *
 */
void AutodiscoveryBlink(BOOL on)
{
    if (on != Autodiscovery.blinking)
    {
      AutoDiscoverySetDeviceBlink(on);
      Autodiscovery.blinking = on;
    }
}

/**
 * \author      Pete McCormick
 *
 * \date        2/22/2006
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Update our local hostnames table
 *              based on a received CIP_HOSTNAME_RESPONSE.
 *
 * \param       HostName
 * \param       IP_Address
 *
 */
void AutodiscoveryHostsUpdate(char * HostName, UINT32 IP_Address)
{
    UINT32 i;
    AUTODISCHOST * pEntry;
    char *tempHostname;
    //UINT32 msec;

    if (!MemMgr || (tempHostname = (char *)MemMgr->GetBlock(MAX_DNS_NAME_LENGTH)) == 0)
      {
        DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_ETHERNET, ERR_ETH_AUTODISC_MEMALLOCFAILED,4);
        return;
      }

    strncpy(tempHostname, HostName, MAX_DNS_NAME_LENGTH);

    // look through existing entries
    for(i=0, pEntry = &Autodiscovery.hostsEntry[0];
        i<AUTODISC_MAX_HOSTS;
        i++, pEntry++)
    {
        if(!pEntry->HostName[0])
        {
            continue;
        }
        if(strcmp(tempHostname, pEntry->HostName) == 0)
        {
            // found a match
            pEntry->IP_Address = IP_Address;
            pEntry->msec = HwGetMsec();
            MemMgr->FreeBlock((unsigned char *)tempHostname);
            return;
        }
    }

#if (AUTODISC_MAX_HOSTS <= 2)

    UINT32 oldestTime = 0xFFFFFFFF;
    AUTODISCHOST *pOldestEntry = &Autodiscovery.hostsEntry[0];

    // need to add a new entry
    for (i = 0, pEntry = &Autodiscovery.hostsEntry[0]; i < AUTODISC_MAX_HOSTS; i++, pEntry++)
    {
        if(pEntry->msec < oldestTime)
        {
            pOldestEntry = pEntry;
            oldestTime = pEntry->msec;
        }
    }
    _strncpy(pOldestEntry->HostName, tempHostname, sizeof(pOldestEntry->HostName));
    pOldestEntry->IP_Address = IP_Address;
    pOldestEntry->msec = HwGetMsec();
    MemMgr->FreeBlock((unsigned char *)tempHostname);

#else  // #if (AUTODISC_MAX_HOSTS > 2)

    // need to add a new entry
    for (i=0, pEntry = &Autodiscovery.hostsEntry[0]; i < AUTODISC_MAX_HOSTS; i++, pEntry++)
    {
        if (!pEntry->HostName[0])
        {
            // found an empty slot!
            strncpy(pEntry->HostName, tempHostname, sizeof(pEntry->HostName));
            pEntry->IP_Address = IP_Address;
            pEntry->msec = HwGetMsec();
            MemMgr->FreeBlock((unsigned char *)tempHostname);
            return;
        }
    }

#ifdef AUTODEBUG
    if(Autodiscovery.debugLevel)
    {
        ErrorPrintf("Can't update autodisc DNS entry for %s\r", tempHostname);
    }
#endif // AUTODEBUG
    MemMgr->FreeBlock((unsigned char *)tempHostname);

#endif  // #if (AUTODISC_MAX_HOSTS <= 2)
}

/**
 * \author      Pete McCormick
 *
 * \date        3/8/2006
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Handle an autodiscovery query.
 *
 * \param       srcIpAddr
 *
 */
void AutodiscoveryQuery(CIP_AUTODISCOVERY_QUERY_MSG * pQuery, UINT32 packetBytes, UINT32 srcIpAddr)
{
    CIP_AUTODISCOVERY_RESPONSE_MSG *queryResp;
    UINT16 CIP_ID = 0;
    MASTER_LIST_ENTRY * pEntry = NULL;
    char *assignerTxt;
    int Flags = 0;
    BOOL hostnameChanged;
    BOOL reboot = 0;

    if (!MemMgr || (assignerTxt = (char *)MemMgr->GetBlock(MAX_DNS_NAME_LENGTH)) == 0)
      {
        DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_ETHERNET, ERR_ETH_AUTODISC_MEMALLOCFAILED,5);
        return;
      } // if (!MemMgr || (assignerTxt = (char *)MemMgr->GetBlock(MAX_DNS_NAME_LENGTH)) == 0)

    if (!MemMgr || (queryResp = (CIP_AUTODISCOVERY_RESPONSE_MSG *)MemMgr->GetBlock(sizeof(CIP_AUTODISCOVERY_RESPONSE_MSG))) == 0)
      {
        DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_ETHERNET, ERR_ETH_AUTODISC_MEMALLOCFAILED,6);
        MemMgr->FreeBlock((unsigned char *)assignerTxt);
        return;
      } // if (!MemMgr || (queryResp = (char *)MemMgr->GetBlock(sizeof(CIP_AUTODISCOVERY_RESPONSE_MSG))) == 0)

    if(packetBytes < sizeof(CIP_AUTODISCOVERY_QUERY_MSG)-sizeof(CIP_HEADER_INFO))
    {
        ErrorPrintf("Received short autodisc query (%u)\n", packetBytes);
        MemMgr->FreeBlock((unsigned char *)assignerTxt);
        MemMgr->FreeBlock((unsigned char *)queryResp);
        return;
    }

#ifdef STATS
    Autodiscovery.queryCnt++;
#endif // STATS
    memset(queryResp, 0, sizeof(CIP_AUTODISCOVERY_RESPONSE_MSG));
    LocalConvertEntireStringToUpper(pQuery->HostName);

    // do endian translations on data in received packet
    pQuery->Flags = BE2NATIVE16(pQuery->Flags);
    pQuery->CIP_ID = BE2NATIVE16(pQuery->CIP_ID);

    // find existing entry in current master list, if any
    if(pQuery->HostName[0])
    {
        pEntry = FindMasterByName(pQuery->HostName);
        if(pEntry)
        {
            CIP_ID = pEntry->CIP_ID;
            queryResp->Flags |= CIP_AUTODIS_RESPONSE_MATCH_FOUND;
        }
		// if host's ip address specified in dot decimal notation
		else
		{
			pEntry = FindMasterByIpString(pQuery->HostName);
			if(pEntry)
			{
				CIP_ID = pEntry->CIP_ID;
				queryResp->Flags |= CIP_AUTODIS_RESPONSE_MATCH_FOUND;
			}
		}
		if(Autodiscovery.enabled)
		{
			AutodiscoveryHostsUpdate(pQuery->HostName, srcIpAddr);
		}

		if(CIPMaxMasters() == 1)
        {
            // If we have at most 1 master,
            // then just replace the entry regardless of hostname
            pEntry = MasterListLockInst(0);
        }
    }

    if(Autodiscovery.enabled && (pQuery->Flags & CIP_AUTODIS_QUERY_CLEAR_IP_TABLE_FLAG))
    {
        if(pQuery->CIP_ID == 0)
        {
            ErrorPrintf("AutoDiscovery: clearing ip table\n");
            // remove all entries
            MasterListWipe();
            queryResp->Flags |= CIP_AUTODIS_RESPONSE_TABLE_CLEARED;
        }
        else if(pQuery->HostName[0])
        {
            // try to clear a specific entry
            if(RemoveEntryFromMasterList(CIP_ID, 0, pQuery->HostName) != CIP_SUCCESS)
            {
                ErrorPrintf("AutoDiscovery: Error removing CIP_ID %x from table\n", CIP_ID);
            }
        }
        else
        {
            // no hostname specified
            ConvertIPAddressToString(srcIpAddr, assignerTxt);
            ErrorPrintf("AutoDiscovery: Required hostname not present in clear from %s\n",
                assignerTxt);
        }
    }

    if(Autodiscovery.enabled && (pQuery->Flags & CIP_AUTODIS_QUERY_SET_IP_ENTRY))
    {
        Flags = CIP_STATIC_MASTER;
        // query MUST contain a hostname!
        if(pQuery->HostName[0])
        {
            strncpy(assignerTxt, pQuery->HostName, MAX_DNS_NAME_LENGTH);
            ErrorPrintf("AutoDiscovery: Accepting CIP_ID %2.2x assignment from %s\n",
                pQuery->CIP_ID, assignerTxt);
            // always store by name!
            //Flags |= CIP_STORE_BY_NAME;
            if(pEntry)
            {
                // there is already an existing entry in the master list
                pEntry->Flags = Flags;
                pEntry->CIP_ID = pQuery->CIP_ID;
                pEntry->IP_Address = 0;
                if(strcmp(pEntry->SiteName,  pQuery->HostName) != 0)
                {
                    // control system hostname changed
                    _strncpy(pEntry->SiteName, pQuery->HostName, MAX_SITE_NAME);
                    hostnameChanged = 1;
                }
                else
                {
                    hostnameChanged = 0;
                }
                GatewayUpdateEntry(pEntry, hostnameChanged);
            }
            else //(!pEntry)
            {
                // was no entry in list - so have to add it now
                // don't save ip address of control system on purpose!
                //if(AddEntryToMasterList(pQuery->CIP_ID, ipAddr, pQuery->HostName, Flags) != CIP_SUCCESS)
                pEntry = AddEntryToMasterList(pQuery->CIP_ID, 0, pQuery->HostName, 0, CIPGetPort(), Flags);
                if(!pEntry)
                {
                    ErrorPrintf("AutoDiscovery: Error adding CIP_ID %2.2x to master list\n",
                        pQuery->CIP_ID);
                }
                else if(AddGatewayEntry(pEntry, pEntry->handle) != 0)
                {
                    ErrorPrintf("AutoDiscovery: Error adding gateway entry for %s\n",
                        pQuery->HostName);
                }
            }
            // save, but ipaddr of entry will always be zero
            MasterListSaveParams();
            queryResp->Flags |= CIP_AUTODIS_RESPONSE_CONFIGURATION_SET;

#ifdef SERIES_3_CONTROL
            // Call function to update the iptable on down stream devices
            if (g_pFuncUpdateDownstreamIpt != NULL)
              g_pFuncUpdateDownstreamIpt();
#endif
        }
        else
        {
            // no hostname in assignment - report error!
            ConvertIPAddressToString(srcIpAddr, assignerTxt);
            ErrorPrintf("AutoDiscovery: Required hostname not present in assignment from %s\n",
                assignerTxt);
        }
    }
    else if(Autodiscovery.enabled && (pQuery->Flags & CIP_AUTODIS_QUERY_START_LIGHT_N_POLL))
    {
        Autodiscovery.srcIpAddr = srcIpAddr;
        AutodiscoveryBlink(1);
        queryResp->Flags |= CIP_AUTODIS_RESPONSE_LIGHT_N_POLL_STARTED;
    }
    else if(Autodiscovery.enabled && (pQuery->Flags & CIP_AUTODIS_QUERY_STOP_LIGHT_N_POLL))
    {
        AutodiscoveryBlink(0);
        queryResp->Flags |= CIP_AUTODIS_RESPONSE_LIGHT_N_POLL_STOPPED;
    }

    // if No IP table entry and respond only if you don't have an IP table entry OR
    // Query response flag is set
    if ((pQuery->Flags & CIP_AUTODIS_QUERY_RESPOND_FLAG) ||
        (Autodiscovery.RespondOnNoIpTableQuery &&  (pQuery->Flags & CIP_AUTODIS_QUERY_QUERY_SEND_IF_IP_TABLE_CLEAR) &&
         pEntry == NULL))
    {
        if(!Autodiscovery.enabled)
        {
            queryResp->Flags = CIP_AUTODIS_RESPONSE_CONFIGURATION_BLOCKED;
        }
        if(queryResp->Flags & CIP_AUTODIS_RESPONSE_TABLE_CLEARED)
        {
            reboot = 1;
        }

        queryResp->Header.PacketType = CIP_AUTODISCOVERY_RESPONSE;
        queryResp->Header.Flags = NATIVE2BE16(queryResp->Header.Flags);
        queryResp->Header.DataLength = (UINT16)NATIVE2BE16(sizeof(CIP_AUTODISCOVERY_RESPONSE_MSG)-sizeof(CIP_HEADER_INFO));

        queryResp->Flags = NATIVE2BE16(queryResp->Flags);
        queryResp->CIP_ID = CIP_ID;
        queryResp->CIP_ID = NATIVE2BE16(queryResp->CIP_ID);
        NetGetHostname(queryResp->LocalHostName, MAX_DNS_NAME_LENGTH);
        AutodiscoveryGetFirmwareName(queryResp->FirmwareName, MAX_DEVICE_NAME_LENGTH);

        AutodiscoverySend(queryResp, sizeof(CIP_AUTODISCOVERY_RESPONSE_MSG), srcIpAddr);

        if(reboot)
        {
            DmConsolePrintf("IP table cleared - Rebooting...\r");
            // make sure any non-volatile data is written to memory
            UpdateSystemNVL();
            DmRebootNotFatal();
        } // if(reboot)
    } // if(pQuery->Flags & CIP_AUTODIS_QUERY_RESPOND_FLAG)

    MemMgr->FreeBlock((unsigned char *)assignerTxt);
    MemMgr->FreeBlock((unsigned char *)queryResp);

}

/**
 * \author      Pete McCormick
 *
 * \date        3/8/2006
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Handle an autodiscovery hostname query.
 *
 * \param       pHostnameQuery
 * \param       packetBytes
 * \param       srcIpAddr
 *
 */
void AutodiscoveryHostnameQuery(CIP_HOSTNAME_QUERY_MSG * pHostnameQuery,
    UINT32 packetBytes, UINT32 srcIpAddr)
{
    CIP_HOSTNAME_RESPONSE_MSG * hostnameResp;
    char * myHostname;

    if (!MemMgr || (myHostname = (char *)MemMgr->GetBlock(MAX_DNS_NAME_LENGTH)) == 0)
      {
        DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_ETHERNET, ERR_ETH_AUTODISC_MEMALLOCFAILED,7);
        return;
      } // if (!MemMgr || (myHostname = (char *)MemMgr->GetBlock(MAX_DNS_NAME_LENGTH)) == 0)

    if (!MemMgr || (hostnameResp = (CIP_HOSTNAME_RESPONSE_MSG *)MemMgr->GetBlock(sizeof(CIP_HOSTNAME_RESPONSE_MSG))) == 0)
      {
        DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_ETHERNET, ERR_ETH_AUTODISC_MEMALLOCFAILED,8);
        MemMgr->FreeBlock((unsigned char *)myHostname);
        return;
      } // if (!MemMgr || (hostnameResp = (char *)MemMgr->GetBlock(sizeof(CIP_HOSTNAME_RESPONSE_MSG))) == 0)


    if(packetBytes < sizeof(CIP_HOSTNAME_QUERY_MSG)-sizeof(CIP_HEADER_INFO))
    {
        ErrorPrintf("Received a short hostname query (%u)\n", packetBytes);
        MemMgr->FreeBlock((unsigned char *)myHostname);
        MemMgr->FreeBlock((unsigned char *)hostnameResp);
        return;
    }
    // someone else is trying to find the ip address for the
    // specified hostname
#ifdef STATS
    Autodiscovery.hostQueryCnt++;
#endif // STATS
    NetGetHostname(myHostname, MAX_DNS_NAME_LENGTH);
    LocalConvertEntireStringToUpper(myHostname);
    if(strcmp(pHostnameQuery->HostName, myHostname) == 0)
    {
        // queried hostname is ours!
        memset(hostnameResp, 0, sizeof(CIP_HOSTNAME_RESPONSE_MSG));

        hostnameResp->Header.PacketType = CIP_HOSTNAME_RESPONSE;
        hostnameResp->Header.Flags = NATIVE2BE16(hostnameResp->Header.Flags);
        hostnameResp->Header.DataLength = (UINT16)NATIVE2BE16(sizeof(CIP_HOSTNAME_RESPONSE_MSG)-sizeof(CIP_HEADER_INFO));

        _strncpy(hostnameResp->HostName, myHostname, MAX_DNS_NAME_LENGTH);
        hostnameResp->IP_Address = NetGetIPAddress();

        // make reply a broadcast
        AutodiscoverySend(hostnameResp, sizeof(CIP_HOSTNAME_RESPONSE_MSG), 0xffffffff);
    } // if(strcmp(pHostnameQuery->HostName, myHostname) == 0)


    MemMgr->FreeBlock((unsigned char *)myHostname);
    MemMgr->FreeBlock((unsigned char *)hostnameResp);

}

/**
 * \author      Pete McCormick
 *
 * \date        3/8/2006
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Handle an autodiscovery hostname response.
 *
 * \param       pHostnameResp
 * \param       packetBytes
 * \param       srcIpAddr
 *
 */
void AutodiscoveryHostnameResponse(CIP_HOSTNAME_RESPONSE_MSG * pHostnameResp,
    UINT32 packetBytes, UINT32 srcIpAddr)
{
    char ipAddrTxt[32];
    UINT32 ipAddr;

    if(packetBytes < sizeof(CIP_HOSTNAME_RESPONSE_MSG)-sizeof(CIP_HEADER_INFO))
    {
        ErrorPrintf("Received a short hostname response (%u)\n", packetBytes);
        return;
    }
    // received a hostname response from someone else
#ifdef STATS
    Autodiscovery.hostRespCnt++;
#endif // STATS
    AutodiscoveryHostnameRespEvent();

    LocalConvertEntireStringToUpper(pHostnameResp->HostName);
    //ipAddr = BE2NATIVE32(pHostnameResp->IP_Address);
    // MNT - 8/28/2008 - We store IP address in the BE format - we might want to change
    // the table. But for now, just reverse it here
    ipAddr = pHostnameResp->IP_Address;

    AutodiscoveryHostsUpdate(pHostnameResp->HostName, ipAddr);
#ifdef AUTODEBUG
    if(Autodiscovery.debugLevel)
    {
        ConvertIPAddressToString(ipAddr, ipAddrTxt);
        ErrorPrintf("AutoDisc: hostname %s resolved to %s\r",
            pHostnameResp->HostName, ipAddrTxt);
    }
#endif // AUTODEBUG
}

/**
 * \author      Pete McCormick
 *
 * \date        3/8/2006
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Handle an autodiscovery hostname set packet
 *
 * \param       pHostnameSet
 * \param       packetBytes
 * \param       srcIpAddr
 *
 */
void AutodiscoveryHostnameSet(CIP_HOSTNAME_SET_MSG * pHostnameSet, UINT32 packetBytes, UINT32 srcIpAddr)
{
    CIP_HOSTNAME_ACK_MSG hostnameAck;
    char *myHostname;

    if (!MemMgr || (myHostname = (char *)MemMgr->GetBlock(MAX_DNS_NAME_LENGTH)) == 0)
      {
        DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_ETHERNET, ERR_ETH_AUTODISC_MEMALLOCFAILED,9);
        return;
      }

    if(packetBytes < sizeof(CIP_HOSTNAME_SET_MSG)-sizeof(CIP_HEADER_INFO))
    {
        ErrorPrintf("Received a short hostname set(%u)\n", packetBytes);
        MemMgr->FreeBlock((unsigned char *)myHostname);
        return;
    }
    // accept our hostname assignment from the control system
    NetGetHostname(myHostname, MAX_DNS_NAME_LENGTH);
    LocalConvertEntireStringToUpper(myHostname);
    if(strcmp(myHostname, pHostnameSet->HostName) == 0)
    {
        MemMgr->FreeBlock((unsigned char *)myHostname);
        return;
    }
    // hostname sent by control system and our hostname
    // are different!
    if(!NetHostnameIsValid(pHostnameSet->HostName))
    {
        ErrorPrintf("AutoDiscovery: ignoring invalid hostname %s\n",
            pHostnameSet->HostName);
        MemMgr->FreeBlock((unsigned char *)myHostname);
        return;
    }

    NetSetHostname(pHostnameSet->HostName);
    ErrorPrintf("AutoDiscovery: hostname %s assigned\n",
        pHostnameSet->HostName);

    memset(&hostnameAck, 0, sizeof(hostnameAck));

    hostnameAck.Header.PacketType = CIP_HOSTNAME_ACK;
    hostnameAck.Header.Flags = NATIVE2BE16(hostnameAck.Header.Flags);
    hostnameAck.Header.DataLength = NATIVE2BE16(sizeof(hostnameAck) - sizeof(hostnameAck.Header));

    AutodiscoverySend(&hostnameAck, sizeof(hostnameAck), srcIpAddr);
    MemMgr->FreeBlock((unsigned char *)myHostname);
}

/**
 * \author      Pete McCormick
 *
 * \date        11/16/2005
 *
 * \return      INT32
 *
 * \retval      0 : success
 * \retval      anything else : failure
 *
 * \brief       Handle an autodiscovery request.
 *
 * \param       ResponsePtr
 * \param       packetBytes - includes the CIP header!
 * \param       srcIpAddr
 *
 */
INT32 AutodiscoveryReceive(void * pPkt, UINT32 packetBytes, UINT32 srcIpAddr)
{
    INT32 result = 0;
    //UINT32 entryId;
    CIP_PACKET * ResponsePtr = (CIP_PACKET *)pPkt;
    UINT16 dataLength;

#ifdef STATS
    Autodiscovery.rcvPktCnt++;
#endif // STATS

    if(packetBytes < sizeof(ResponsePtr->Header))
    {
        ErrorPrintf("Received a short CIP packet (%u)\n", packetBytes);
        return -1;
    }

    dataLength = BE2NATIVE16(ResponsePtr->Header.DataLength);
    if(packetBytes != dataLength + sizeof(ResponsePtr->Header))
    {
        ErrorPrintf("Invalid header length (%u), pkttype %x, ip addr %d.%d.%d.%d\n", dataLength, ResponsePtr->Header.PacketType,
								srcIpAddr&255, (srcIpAddr>>8)&255, (srcIpAddr>>16)&255, (srcIpAddr>>24)&255);

		return -1;
    }

    if(!Autodiscovery.enabled)
    {
        if(ResponsePtr->Header.PacketType == CIP_AUTODISCOVERY_QUERY)
        {
            return -1;
        }
    }
#ifdef AUTODEBUG
    if(Autodiscovery.debugLevel > 2)
    {
        ErrorPrintf("Rec'd autodisc pkt %u\r",
                ResponsePtr->Header.PacketType);
    }
#endif // AUTODEBUG
    switch(ResponsePtr->Header.PacketType)
    {
        case CIP_AUTODISCOVERY_QUERY:
            AutodiscoveryQuery((CIP_AUTODISCOVERY_QUERY_MSG *)ResponsePtr, packetBytes, srcIpAddr);
            break;

        case CIP_HOSTNAME_QUERY:
            AutodiscoveryHostnameQuery((CIP_HOSTNAME_QUERY_MSG *)ResponsePtr, packetBytes, srcIpAddr);
            break;

        case CIP_HOSTNAME_RESPONSE:
            AutodiscoveryHostnameResponse((CIP_HOSTNAME_RESPONSE_MSG *)ResponsePtr, packetBytes, srcIpAddr);
            break;

        case CIP_HOSTNAME_SET:
            AutodiscoveryHostnameSet((CIP_HOSTNAME_SET_MSG *)ResponsePtr, packetBytes, srcIpAddr);
            break;

        case CIP_HOSTNAME_ACK:
            //ErrorPrintf("Received an unexpected hostname ack\n");
            // should not get this
            break;

        default:
            break;
    }

    return result;
}

/**
 * \author      Pete McCormick
 *
 * \date        11/16/2005
 *
 * \return      BOOL
 *
 * \retval      1: The autodiscovery LED should be blinking
 * \retval      0: The autodiscovery LED should not be blinking
 *
 * \brief       Returns what the autodiscovery LED should be doing.
 *
 * \param       void
 *
 */
BOOL AutodiscoveryIsBlinking(void)
{
    return Autodiscovery.blinking;
}

/**
 * \author      Pete McCormick
 *
 * \date        11/16/2005
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Do something when the autodiscovery setup button is pressed.
 *
 * \param       void
 *
 */
void AutodiscoverySetupButtonPressed(void)
{
  if(Autodiscovery.blinking)
  {
    CIP_AUTODISCOVERY_RESPONSE_MSG *queryResp;

    if (!MemMgr || (queryResp = (CIP_AUTODISCOVERY_RESPONSE_MSG *)MemMgr->GetBlock(sizeof(CIP_AUTODISCOVERY_RESPONSE_MSG))) == 0)
    {
      DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_ETHERNET, ERR_ETH_AUTODISC_MEMALLOCFAILED,10);
      return;
    } // if (!MemMgr || (queryResp = (char *)MemMgr->GetBlock(sizeof(CIP_AUTODISCOVERY_RESPONSE_MSG))) == 0)


    memset(queryResp, 0, sizeof(CIP_AUTODISCOVERY_RESPONSE_MSG));

    queryResp->Header.PacketType = CIP_AUTODISCOVERY_RESPONSE;
    queryResp->Header.Flags = NATIVE2BE16(queryResp->Header.Flags);
    queryResp->Header.DataLength = (UINT16)NATIVE2BE16(sizeof(CIP_AUTODISCOVERY_RESPONSE_MSG)-sizeof(CIP_HEADER_INFO));

    queryResp->Flags = CIP_AUTODIS_RESPONSE_SETUP_BUTTON_PRESSED;
    queryResp->Flags = NATIVE2BE16(queryResp->Flags);

    NetGetHostname(queryResp->LocalHostName, MAX_DNS_NAME_LENGTH);
    AutodiscoveryGetFirmwareName(queryResp->FirmwareName, MAX_DEVICE_NAME_LENGTH);

    AutodiscoverySend((CIP_PACKET *)queryResp, sizeof(CIP_AUTODISCOVERY_RESPONSE_MSG), Autodiscovery.srcIpAddr);

    MemMgr->FreeBlock((unsigned char *)queryResp);

  } // if(Autodiscovery.blinking)

}

/**
 * \author      Pete McCormick
 *
 * \date        3/8/2006
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Clear the hosts table.
 *
 * \param       void
 *
 */
void AutodiscoveryClearCache(void)
{
    // get semaphore?
    memset(Autodiscovery.hostsEntry, 0, AUTODISC_MAX_HOSTS*sizeof(AUTODISCHOST));
    // release semaphore?
}

/**
 * \author      Pete McCormick
 *
 * \date        3/8/2006
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Do something when the ethernet link goes up
 *
 * \param       void
 *
 */
void AutodiscoveryEthernetLinkUp(UINT32 inst)
{
    INT32 status;

    status = EthCheckInitDone();

    if((NetGetIPAddress() !=0) && (status == 0))
    {
        AutodiscoverySendHostnameResp();
    }
}

/**
 * \author      Pete McCormick
 *
 * \date        3/8/2006
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Do something when the ethernet link goes down
 *
 * \param       void
 *
 */
void AutodiscoveryEthernetLinkDown(UINT32 inst)
{
    AutodiscoveryClearCache();
}

/**
 * \author      Pete McCormick
 *
 * \date        3/8/2006
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Do something when our ip address changes
 *
 * \param       void
 *
 * \note        disabled for now since we always reboot after an IP address change
 */
void AutodiscoveryIPAddressChange(UINT32 inst)
{
    //INT32 status;
    //
    //status = EthCheckInitDone();
    //
    //if(GetIPAddress() && (status == 0))
    //{
    //    AutodiscoverySendHostnameResp();
    //}
}

/**
 * \author      Pete McCormick
 *
 * \date        3/8/2006
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Do something when our hostname changes
 *
 * \param       void
 *
 */
void AutodiscoveryHostnameChange(void)
{
    INT32 status;

    status = EthCheckInitDone();

    if(NetGetIPAddress() && (status == 0))
    {
        AutodiscoverySendHostnameResp();
    }
}

/**
 * \author      Pete McCormick
 *
 * \date        3/8/2006
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Do something when CIP protocol is up and running
 *              (UDP socket and task created)
 *
 * \param       void
 *
 */
void AutodiscoveryCIPReady(void)
{
    INT32 status;

    status = EthCheckInitDone();

    if(NetGetIPAddress() && (status == 0))
    {
        AutodiscoverySendHostnameResp();
    }
}

void AutodiscoverySetEnabled(BOOL set)
{
    Autodiscovery.enabled = set;
}
/**
 * \author      Larry Salant
 * \brief       Special mode for DLP projector "controllers".  they will do an query
 *              for "new" devices.  Only devices without an IP Table entry should respond
 * \date        9/17/2009
 * \param       bool - ON = respond to query
 * \return      void
 * \retval      void
 *
 */
void AutodiscoveryRespondToNoIpTableQuery(BOOL set)
{
    Autodiscovery.RespondOnNoIpTableQuery = set;
}

/**
 * \author      Pete McCormick
 *
 * \date        2/23/2006
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Function to output the current hosts table to the console.
 *
 * \param       void
 *
 */
void AutodiscoveryPrintHostsTable(void)
{
    UINT32 i;
    AUTODISCHOST * pEntry;
    char ipAddrTxt[32];

    DmConsolePrintf("Autodiscovery Hosts Table:\r");

    // get semaphore?
    for(i=0, pEntry = &Autodiscovery.hostsEntry[0];
        i<AUTODISC_MAX_HOSTS;
        i++, pEntry++)
    {
        if(!pEntry->HostName[0])
        {
            continue;
        }
        ConvertIPAddressToString(pEntry->IP_Address, ipAddrTxt);
        DmConsolePrintf("%-32s%-40s\r", ipAddrTxt, pEntry->HostName);
    }
    DmConsolePrintf("\r");
    // release semaphore?
}

void AutodiscoverySetDebugLevel(UINT32 level)
{
#ifdef AUTODEBUG
    Autodiscovery.debugLevel = level;
#endif //AUTODEBUG
}

UINT32 AutodiscoveryGetDebugLevel(void)
{
#ifdef AUTODEBUG
	return Autodiscovery.debugLevel;
#else
	return 0;
#endif // AUTODEBUG
}
